/*
 * Copyright 2021-2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.events.java;

import org.apiguardian.api.API;
import org.opentest4j.reporting.events.api.ChildElement;
import org.opentest4j.reporting.events.api.Context;
import org.opentest4j.reporting.events.core.Result;
import org.opentest4j.reporting.schema.Namespace;
import org.opentest4j.reporting.schema.QualifiedName;

import java.io.PrintWriter;
import java.io.StringWriter;

import static org.apiguardian.api.API.Status.EXPERIMENTAL;

/**
 * The {@code throwable} element of the Java namespace.
 *
 * @since 0.1.0
 */
@API(status = EXPERIMENTAL, since = "0.1.0")
public class Throwable extends ChildElement<Result, Throwable> {

	/**
	 * Qualified name of the {@code throwable} element.
	 */
	public static final QualifiedName ELEMENT = QualifiedName.of(Namespace.REPORTING_JAVA, "throwable");

	/**
	 * Qualified name of the {@code type} attribute.
	 */
	public static final QualifiedName TYPE = QualifiedName.of(Namespace.REPORTING_JAVA, "type");
	private static final QualifiedName ASSERTION_ERROR = QualifiedName.of(Namespace.REPORTING_JAVA, "assertionError");

	Throwable(Context context) {
		super(context, Throwable.ELEMENT);
	}

	Throwable withThrowable(java.lang.Throwable throwable) {
		withAttribute(TYPE, throwable.getClass().getName());
		withAttribute(ASSERTION_ERROR, String.valueOf(throwable instanceof AssertionError));
		StringWriter stringWriter = new StringWriter();
		try (PrintWriter writer = new PrintWriter(stringWriter)) {
			throwable.printStackTrace(writer);
		}
		withCDataSection(stringWriter.toString());
		return this;
	}
}
