#!/usr/bin/env python3

import os
import xml.dom.minidom
from dataclasses import dataclass

def create_xml_file_minidom(filename, listunit):
    """Creates an XML file using xml.dom.minidom."""

    doc = xml.dom.minidom.Document()
    
    root = doc.createElement("xs:schema")
    root.setAttribute("xmlns:xs", "http://www.w3.org/2001/XMLSchema")
    root.setAttribute("xmlns:scale", "http://docs.oasis-open.org/ns/emix/2011/06/siscale")
    root.setAttribute("targetNamespace", "http://docs.oasis-open.org/ns/emix/2011/06/siscale")
    root.setAttribute("elementFormDefault", "qualified")

    root.setAttribute("attributeFormDefault", "unqualified")
    root.setAttribute("version", "1.0")
    doc.appendChild(root)

    element1 = doc.createElement("xs:element")
    element1.setAttribute("name", "siScaleCode")
    element1.setAttribute("type", "scale:SiScaleCodeType")
    root.appendChild(element1)

    content1 = doc.createElement("xs:simpleType")
    content1.setAttribute("name", "SiScaleCodeType")
    root.appendChild(content1)

    content1_1 = doc.createElement("xs:annotation")
    content1.appendChild(content1_1)

    content1_2 = doc.createElement("xs:documentation")
    content1_2.appendChild(doc.createTextNode("Scale based on representations of SI scale as expressed in the unit multipliers"))
    content1_1.appendChild(content1_2);

    content1_3 = doc.createElement("xs:documentation")
    content1_3.setAttribute("xml:lang", "en")
    content1_3.appendChild(doc.createTextNode("enumeration"))
    content1_1.appendChild(content1_3);

    content2 = doc.createElement("xs:restriction")
    content2.setAttribute("base", "xs:string")
    content1.appendChild(content2)
 
    for unit1 in listunit:
        unitdata1 = doc.createElement("xs:enumeration")
        unitdata1.setAttribute("value", unit1.symbol)
        unitdata2 = doc.createElement("xs:annotation")
        unitdata1.appendChild(unitdata2)
        unitdata3 = doc.createElement("xs:documentation")
        unitdata2.appendChild(unitdata3)
        unitdata3.appendChild(doc.createTextNode("%s 10**%d"%(unit1.name, unit1.base10)))
        unitdata4 = doc.createElement("xs:appinfo")
        unitdata4.appendChild(doc.createTextNode("%d"%(unit1.base10)))
        unitdata2.appendChild(unitdata3)
        unitdata2.appendChild(unitdata4)
        content2.appendChild(unitdata1)
    
    xml_str = doc.toprettyxml(indent="        ")

    with open(filename, "w", encoding="utf-8") as file:
        file.write(xml_str)

    print(f"XML file created successfully: {filename}")

@dataclass
class SIUnit:
    name: str
    symbol: str
    base10: int
    
def main():
    siunits = [
        SIUnit("Quecto", "q", -30),
        SIUnit("Ronto", "r", -27),
        SIUnit("Yocto", "y", -24),
        SIUnit("Zepto", "z", -21),
        SIUnit("Atto", "a", -18),
        SIUnit("Femto", "f", -15),
        SIUnit("Pico", "p", -12),
        SIUnit("Nano", "n", -9),
        SIUnit("Micro", "micro", -6),
        SIUnit("Milli", "m", -3),
        SIUnit("Centi", "c", -2),
        SIUnit("Deci", "d", -1),
        SIUnit("Deca", "da", 1),
        SIUnit("Hecto", "h", 2),
        SIUnit("Kilo", "k", 3),
        SIUnit("Mega", "M", 6),
        SIUnit("Giga", "G", 9),
        SIUnit("Tera", "T", 12),
        SIUnit("Peta", "P", 15),
        SIUnit("Exa", "E", 18),
        SIUnit("Zetta", "Z", 21),
        SIUnit("Yotta", "Y", 24),
        SIUnit("Ronna", "R", 27),
        SIUnit("Quetta", "Q", 30),
        SIUnit("Native Scale", "none", 0)
    ]  
    
    file_path_xml = "oadr_siscale_20b.xsd"

    create_xml_file_minidom(file_path_xml, siunits)
    
            
if __name__ == "__main__":
    main()
