% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/methods_lavaan.R, R/methods_psych.R
\name{model_parameters.lavaan}
\alias{model_parameters.lavaan}
\alias{model_parameters.principal}
\title{Parameters from PCA, FA, CFA, SEM}
\usage{
\method{model_parameters}{lavaan}(
  model,
  ci = 0.95,
  standardize = FALSE,
  component = c("regression", "correlation", "loading", "defined"),
  keep = NULL,
  drop = NULL,
  verbose = TRUE,
  ...
)

\method{model_parameters}{principal}(
  model,
  sort = FALSE,
  threshold = NULL,
  labels = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{model}{Model object.}

\item{ci}{Confidence Interval (CI) level. Default to \code{0.95} (\verb{95\%}).}

\item{standardize}{Return standardized parameters (standardized coefficients).
Can be \code{TRUE} (or \code{"all"} or \code{"std.all"}) for standardized
estimates based on both the variances of observed and latent variables;
\code{"latent"} (or \code{"std.lv"}) for standardized estimates based
on the variances of the latent variables only; or \code{"no_exogenous"}
(or \code{"std.nox"}) for standardized estimates based on both the
variances of observed and latent variables, but not the variances of
exogenous covariates. See \code{lavaan::standardizedsolution} for details.}

\item{component}{What type of links to return. Can be \code{"all"} or some of
\code{c("regression", "correlation", "loading", "variance", "mean")}.}

\item{keep}{Character containing a regular expression pattern that
describes the parameters that should be included (for \code{keep}) or excluded
(for \code{drop}) in the returned data frame. \code{keep} may also be a
named list of regular expressions. All non-matching parameters will be
removed from the output. If \code{keep} is a character vector, every parameter
name in the \emph{"Parameter"} column that matches the regular expression in
\code{keep} will be selected from the returned data frame (and vice versa,
all parameter names matching \code{drop} will be excluded). Furthermore, if
\code{keep} has more than one element, these will be merged with an \code{OR}
operator into a regular expression pattern like this: \code{"(one|two|three)"}.
If \code{keep} is a named list of regular expression patterns, the names of the
list-element should equal the column name where selection should be
applied. This is useful for model objects where \code{model_parameters()}
returns multiple columns with parameter components, like in
\code{\link[=model_parameters.lavaan]{model_parameters.lavaan()}}. Note that the regular expression pattern
should match the parameter names as they are stored in the returned data
frame, which can be different from how they are printed. Inspect the
\verb{$Parameter} column of the parameters table to get the exact parameter
names.}

\item{drop}{See \code{keep}.}

\item{verbose}{Toggle warnings.}

\item{...}{Arguments passed to or from other methods.}

\item{sort}{Sort the loadings.}

\item{threshold}{A value between 0 and 1 indicates which (absolute) values
from the loadings should be removed. An integer higher than 1 indicates the
n strongest loadings to retain. Can also be \code{"max"}, in which case it will
only display the maximum loading per variable (the most simple structure).}

\item{labels}{A character vector containing labels to be added to the
loadings data. Usually, the question related to the item.}
}
\value{
A data frame of indices or loadings.
}
\description{
Format structural models from the \strong{psych} or \strong{FactoMineR} packages. There
is a \code{summary()} method for the returned output from \code{model_parameters()}, to
show further information. See 'Examples'.
}
\details{
For the structural models obtained with \strong{psych}, the following indices
are present:
\itemize{
\item \strong{Complexity} (\cite{Hoffman's, 1978; Pettersson and Turkheimer,
2010}) represents the number of latent components needed to account for
the observed variables. Whereas a perfect simple structure solution has a
complexity of 1 in that each item would only load on one factor, a
solution with evenly distributed items has a complexity greater than 1.
\item \strong{Uniqueness} represents the variance that is 'unique' to the
variable and not shared with other variables. It is equal to \verb{1 – communality} (variance that is shared with other variables). A uniqueness
of \code{0.20} suggests that \verb{20\%} or that variable's variance is not shared
with other variables in the overall factor model. The greater 'uniqueness'
the lower the relevance of the variable in the factor model.
\item \strong{MSA} represents the Kaiser-Meyer-Olkin Measure of Sampling
Adequacy (\cite{Kaiser and Rice, 1974}) for each item. It indicates
whether there is enough data for each factor give reliable results for the
PCA. The value should be > 0.6, and desirable values are > 0.8
(\cite{Tabachnick and Fidell, 2013}).
}
}
\note{
There is also a
\href{https://easystats.github.io/see/articles/parameters.html}{\code{plot()}-method}
for \code{lavaan} models implemented in the
\href{https://easystats.github.io/see/}{\strong{see}-package}.
}
\examples{
\dontshow{if (all(insight::check_if_installed(c("psych", "lavaan"), quietly = TRUE))) withAutoprint(\{ # examplesIf}
library(parameters)
\donttest{
# Principal Component Analysis (PCA) ---------
data(attitude)
pca <- psych::principal(attitude)
model_parameters(pca)
summary(model_parameters(pca))

pca <- psych::principal(attitude, nfactors = 3, rotate = "none")
model_parameters(pca, sort = TRUE, threshold = 0.2)

principal_components(attitude, n = 3, sort = TRUE, threshold = 0.2)


# Exploratory Factor Analysis (EFA) ---------
efa <- psych::fa(attitude, nfactors = 3)
model_parameters(efa,
  threshold = "max", sort = TRUE,
  labels = as.character(1:ncol(attitude))
)


# Omega ---------
data(mtcars)
omega <- psych::omega(mtcars, nfactors = 3, plot = FALSE)
params <- model_parameters(omega)
params
summary(params)
}


# lavaan -------------------------------------
# Confirmatory Factor Analysis (CFA) ---------

data(HolzingerSwineford1939, package = "lavaan")
structure <- " visual  =~ x1 + x2 + x3
               textual =~ x4 + x5 + x6
               speed   =~ x7 + x8 + x9 "
model <- lavaan::cfa(structure, data = HolzingerSwineford1939)
model_parameters(model)
model_parameters(model, standardize = TRUE)

# filter parameters
model_parameters(
  model,
  parameters = list(
    To = "^(?!visual)",
    From = "^(?!(x7|x8))"
  )
)

# Structural Equation Model (SEM) ------------

data(PoliticalDemocracy, package = "lavaan")
structure <- "
  # latent variable definitions
    ind60 =~ x1 + x2 + x3
    dem60 =~ y1 + a*y2 + b*y3 + c*y4
    dem65 =~ y5 + a*y6 + b*y7 + c*y8
  # regressions
    dem60 ~ ind60
    dem65 ~ ind60 + dem60
  # residual correlations
    y1 ~~ y5
    y2 ~~ y4 + y6
    y3 ~~ y7
    y4 ~~ y8
    y6 ~~ y8
"
model <- lavaan::sem(structure, data = PoliticalDemocracy)
model_parameters(model)
model_parameters(model, standardize = TRUE)
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item Kaiser, H.F. and Rice. J. (1974). Little jiffy, mark iv. Educational and
Psychological Measurement, 34(1):111–117
\item Pettersson, E., and Turkheimer, E. (2010). Item selection, evaluation, and
simple structure in personality data. Journal of research in personality,
44(4), 407-420.
\item Revelle, W. (2016). How To: Use the psych package for Factor Analysis and
data reduction.
\item Tabachnick, B. G., and Fidell, L. S. (2013). Using multivariate statistics
(6th ed.). Boston: Pearson Education.
\item Rosseel Y (2012). lavaan: An R Package for Structural Equation
Modeling. Journal of Statistical Software, 48(2), 1-36.
\item Merkle EC , Rosseel Y (2018). blavaan: Bayesian Structural Equation
Models via Parameter Expansion. Journal of Statistical Software, 85(4),
1-30. http://www.jstatsoft.org/v85/i04/
}
}
